package theorycrafter.utils

import androidx.compose.runtime.Composable
import androidx.compose.ui.window.AwtWindow
import compose.utils.LocalWindow
import java.awt.FileDialog
import java.awt.Frame
import java.io.File
import java.io.FilenameFilter


/**
 * Shows a native file picker dialog for picking a file to read from.
 */
@Composable
fun NativeReadFilePickerDialog(
    title: String,
    directory: File? = null,
    filenameFilter: FilenameFilter? = null,
    onCompletedSelecting: (file: File?) -> Unit
) {
    NativeFilePickerDialog(
        title = title,
        file = null,
        directory = directory?.absolutePath,
        mode = FileDialog.LOAD,
        filenameFilter = filenameFilter,
        onCompletedSelecting = onCompletedSelecting
    )
}


/**
 * Shows a native file picker dialog for picking a file to write to.
 */
@Composable
fun NativeWriteFilePickerDialog(
    title: String,
    suggestedFilename: String? = null,
    filenameFilter: FilenameFilter? = null,
    onCompletedSelecting: (file: File?) -> Unit
) {
    NativeFilePickerDialog(
        title = title,
        file = suggestedFilename,
        directory = null,
        mode = FileDialog.SAVE,
        filenameFilter = filenameFilter,
        onCompletedSelecting = { file ->
            onCompletedSelecting(file)
        }
    )
}


/**
 * Shows a native file picker dialog for picking a file to write to.
 */
@Composable
fun NativeWriteFilePickerDialog(
    title: String,
    suggestedFile: File?,
    filenameFilter: FilenameFilter? = null,
    onCompletedSelecting: (file: File?) -> Unit
) {
    NativeFilePickerDialog(
        title = title,
        file = suggestedFile?.name,
        directory = suggestedFile?.parentFile?.absolutePath,
        mode = FileDialog.SAVE,
        filenameFilter = filenameFilter,
        onCompletedSelecting = { file ->
            onCompletedSelecting(file)
        }
    )
}


/**
 * Shows a native file picker dialog.
 */
@Composable
fun NativeFilePickerDialog(
    title: String,
    file: String?,
    directory: String?,
    mode: Int,
    filenameFilter: FilenameFilter? = null,
    onCompletedSelecting: (file: File?) -> Unit
) {
    val parent = LocalWindow.current as Frame
    AwtWindow(
        create = {
            object : FileDialog(parent, title, mode) {
                override fun setVisible(value: Boolean) {
                    super.setVisible(value)
                    if (value)
                        onCompletedSelecting(files.firstOrNull())
                }
            }.also {
                it.directory = directory
                it.file = file
                it.filenameFilter = filenameFilter
                it.isMultipleMode = false
                it.pack()
                it.setLocationRelativeTo(parent)
            }
        },
        dispose = FileDialog::dispose
    )
}


/**
 * Returns a [FilenameFilter] that only accepts files with the given list of extensions.
 */
fun extensionFilenameFilter(ignoreCase: Boolean, vararg extensions: String) = FilenameFilter { _, name ->
    (name != null) && extensions.any { name.endsWith(".$it", ignoreCase = ignoreCase) }
}