package theorycrafter.ui.settings

import androidx.compose.foundation.layout.*
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import compose.utils.rememberCenteredWindowState
import compose.widgets.VerticallyScrollableContent
import theorycrafter.LocalTheorycrafterWindowManager
import theorycrafter.Theorycrafter
import theorycrafter.TheorycrafterContext.settings
import theorycrafter.TheorycrafterWindow
import theorycrafter.TheorycrafterWindowInfo
import theorycrafter.ui.AboutDialogContent
import theorycrafter.ui.FitExportOptionsEditor
import theorycrafter.ui.Icons
import theorycrafter.ui.TheorycrafterTheme
import theorycrafter.ui.widgets.PaneDescriptor
import theorycrafter.ui.widgets.Panes


/**
 * The window for editing settings.
 */
@Composable
fun SettingsWindow() {
    val windowManager = LocalTheorycrafterWindowManager.current
    val state = windowManager.settingsWindowState
    if (state !is SettingsWindowState.Open)
        return

    val windowState = rememberCenteredWindowState(
        baseWindowState = windowManager.mainWindowState,
        TheorycrafterTheme.sizes.settingsWindowDefaultSize
    )
    TheorycrafterWindow(
        title = "Settings",
        onCloseRequest = windowManager::closeSettingsWindow,
        state = windowState,
    ) {
        SettingsWindowContent(state)

        LaunchedEffect(state, window) {
            state.window = window
        }
    }
}


/**
 * The state of the settings window.
 */
@Stable
sealed interface SettingsWindowState {


    /**
     * The window is closed.
     */
    @Stable
    data object Closed: SettingsWindowState


    /**
     * The window is open, showing the given pane.
     */
    @Stable
    class Open(pane: SettingsPane): TheorycrafterWindowInfo(), SettingsWindowState {


        /**
         * The displayed pane.
         */
        var pane: SettingsPane by mutableStateOf(pane)


    }


}


/**
 * The content of the settings window.
 */
@Composable
private fun SettingsWindowContent(state: SettingsWindowState.Open) {
    Panes(
        paneDescriptors = SettingsPane.entries,
        paneDescriptorsListModifier = Modifier.width(TheorycrafterTheme.sizes.settingsPaneListWidth),
        paneContent = {
            if (it.verticallyScrollable) {
                VerticallyScrollableContent {
                    it.content()
                }
            } else {
                it.content()
            }
        },
        selectedPane = state.pane,
        setSelectedPane = { state.pane = it },
    )
}


/**
 * The panes in the settings window.
 */
enum class SettingsPane(
    override val title: String,
    override val icon: @Composable (Modifier) -> Unit,
    val content: @Composable () -> Unit,
    val verticallyScrollable: Boolean
): PaneDescriptor {

    Appearance(
        title = "Appearance",
        icon = { Icons.AppearanceSettings(it) },
        content = { AppearanceSettingsPane() },
        verticallyScrollable = true
    ),
    FitExportOptions(
        title = "Fit Export",
        icon = { Icons.Share(it) },
        content = { FitExportOptionsPane() },
        verticallyScrollable = true
    ),
    SkillSets(
        title = "Skill Sets",
        icon = { Icons.SkillSetSettings(it) },
        content = { SkillSetsSettings() },
        verticallyScrollable = false,
    ),
    Tournaments(
        title = "Tournaments",
        icon = { Icons.Tournaments(it) },
        content = { TournamentSettingsPane() },
        verticallyScrollable = true
    ),
    Prices(
        title = "Prices",
        icon = { Icons.Prices(it) },
        content = { PriceSettingsPane() },
        verticallyScrollable = true
    ),
    TrackingConsent(
        title = "Tracking Consent",
        icon = { Icons.TrackingConsent(it) },
        content = { AnalyticsConsentSettingsPane() },
        verticallyScrollable = true
    ),
    SoftwareUpdates(
        title = "${Theorycrafter.AppName} Updates",
        icon = { Icons.SoftwareUpdates(it) },
        content = { SoftwareUpdatesPane() },
        verticallyScrollable = false
    ),
    About(
        title = "About ${Theorycrafter.AppName}",
        icon = { Icons.AboutSettings(it) },
        content = { AboutPane() },
        verticallyScrollable = true
    );


    companion object {

        val Default: SettingsPane
            get() = Appearance

    }


}


/**
 * The "About" pane.
 */
@Composable
private fun AboutPane() {
    Box(
        modifier = Modifier.fillMaxWidth(),
        contentAlignment = Alignment.TopCenter
    ) {
        AboutDialogContent(
            Modifier
                .width(TheorycrafterTheme.sizes.aboutDialogWidth)
                .padding(TheorycrafterTheme.spacing.xxxlarge)
                .padding(top = TheorycrafterTheme.spacing.xxxlarge)
        )
    }
}


/**
 * The pane to configure fit export options used by the regular "Copy Fit" action.
 */
@Composable
private fun FitExportOptionsPane() {
    val settings = settings
    Column(
        modifier = Modifier.padding(TheorycrafterTheme.spacing.edgeMargins),
        verticalArrangement = Arrangement.spacedBy(TheorycrafterTheme.spacing.large)
    ) {
        Text("When copying a fit to the clipboard, include:")
        FitExportOptionsEditor(
            exportOptionSettings = settings.regularFitExportOptions,
            onExportOptionsChanged = { settings.regularFitExportOptions = it },
        )
    }
}
