package theorycrafter.fitting

import androidx.compose.runtime.Stable
import eve.data.Attributes
import eve.data.DamageType
import eve.data.EnvironmentType
import eve.data.utils.ValueByEnum
import eve.data.utils.valueByEnum


/**
 * An instance of an environment effect, e.g. wormwhole, abyssal weather etc.
 */
@Stable
class Environment internal constructor(


    /**
     * The fit this booster is part of.
     */
    override val fit: Fit,


    /**
     * The context [Attributes].
     */
    attributes: Attributes,


    /**
     * The environment type.
     */
    type: EnvironmentType


): EveItem<EnvironmentType>(attributes, type), FitItemWithEnabledState {


    override val enabledState = EnabledState()


    /**
     * The bonus to shield resistances.
     */
    val shieldResistanceBonus: ValueByEnum<DamageType, AttributeProperty<Double>?> by lazy {
        valueByEnum { damageType ->
            propertyOrNull(attributes.shieldResistanceBonus[damageType])
        }
    }


    /**
     * The bonus to armor resistances.
     */
    val armorResistanceBonus: ValueByEnum<DamageType, AttributeProperty<Double>?> by lazy {
        valueByEnum { damageType ->
            propertyOrNull(attributes.armorResistanceBonus[damageType])
        }
    }


    /**
     * The multiplier of shield hitpoints.
     */
    val shieldHitpointsMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.shieldHitpointsMultiplier)


    /**
     * The multiplier of armor hitpoints.
     */
    val armorHitpointsMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.armorHpMultiplier)


    /**
     * The multiplier of signature radius.
     */
    val signatureRadiusMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.signatureRadiusMultiplier)


    /**
     * The multiplier of capacitor recharge time.
     */
    val rechargeRateMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.rechargeRateMultiplier)


    /**
     * The multiplier of some ewar strength.
     */
    val energyWarfareStrengthMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.energyWarfareStrengthMultiplier)


    /**
     * The multiplier of agility (inertia).
     */
    val agilityMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.agilityMultiplier)


    /**
     * The multiplier of maximum targeting range.
     */
    val maxTargetRangeMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.maxTargetRangeMultiplier)


    /**
     * The multiplier of missile velocity.
     */
    val missileVelocityMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.missileVelocityMultiplier)


    /**
     * The multiplier of maximum ship velocity.
     */
    val maxVelocityMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.maxVelocityMultiplier)


    /**
     * The multiplier of explosion velocity.
     */
    val explosionVelocityMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.explosionVelocityMultiplier)


    /**
     * The multiplier of explosion radius.
     */
    val explosionRadiusMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.explosionRadiusMultiplier)


    /**
     * The multiplier of stasis webifier strength.
     */
    val stasisWebStrengthMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.stasisWebStrengthMultiplier)


    /**
     * The multiplier of local armor repair amount.
     */
    val armorRepairAmountMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.armorRepairAmountMultiplier)


    /**
     * The multiplier of remote armor repair amount.
     */
    val armorRemoteRepairAmountMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.armorRemoteRepairAmountMultiplier)


    /**
     * The multiplier of local shield boost amount.
     */
    val shieldBoostAmountMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.shieldBoostAmountMultiplier)


    /**
     * The multiplier of remote shield boost amount.
     */
    val shieldRemoteBoostAmountMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.shieldRemoteBoostAmountMultiplier)


    /**
     * The multiplier of capacitor capacity.
     */
    val capacitorCapacityMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.capacitorCapacityMultiplier)


    /**
     * The multiplier of energy transfer amount.
     */
    val energyTransferAmountMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.energyTransferAmountMultiplier)


    /**
     * The multiplier of tracking speed.
     */
    val trackingSpeedMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.trackingSpeedMultiplier)


    /**
     * The multiplier of weapon damage.
     */
    val damageMultiplierMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.damageMultiplierMultiplier)


    /**
     * The multiplier of target painter effectiveness.
     */
    val targetPainterEffectivenessMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.targetPainterEffectivenessMultiplier)


    /**
     * The multiplier of heat damage taken by overloaded modules.
     */
    val heatDamageMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.heatDamageMultiplier)


    /**
     * The multiplier of overload effect.
     */
    val overloadBonusMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.overloadBonusMultiplier)


    /**
     * The multiplier of smartbomb range.
     */
    val smartbombRangeMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.smartbombRangeMultiplier)


    /**
     * The multiplier of smartbomb damage and regular bomb damage/effectiveness.
     */
    val bombEffectivenessMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.bombEffectivenessMultiplier)


    /**
     * The multiplier of small weapons damage.
     */
    val smallWeaponsDamageMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.smallWeaponDamageMultiplier)


    /**
     * The bonus (or penalty) to virus coherence.
     */
    val virusCoherenceBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.virusCoherenceBonus)


    /**
     * The bonus (or penalty) to scan probe strength.
     */
    val scanProbeStrengthBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.scanProbeStrengthBonus)


    /**
     * Whether cloaking is disallowed in this environment.
     */
    val isCloakingDisallowed: Boolean
        get() = propertyOrNull(attributes.disallowCloaking)?.value ?: false


    /**
     * The bonus (or penalty) to scan resolution.
     */
    val scanResolutionBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.scanResolutionBonus)


    /**
     * The multiplier of mining laser duration.
     */
    val miningDurationMultiplier: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.miningDurationMultiplier)


    /**
     * The bonus (or penalty) to warp speed.
     */
    val warpSpeedBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.warpSpeedBonus)


    /**
     * The bonus (or penalty) to local shield booster duration.
     */
    val shieldBoosterDurationBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.shieldBoosterDurationBonus)


    /**
     * The bonus (or penalty) to local shield booster duration.
     */
    val armorRepairerDurationBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.armorRepairerDurationBonus)


    /**
     * The bonus (or penalty) to turret optimal range.
     */
    val turretOptimalRangeBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.optimalRangeBonus)


    /**
     * The bonus (or penalty) to turret falloff.
     */
    val turretFalloffBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.falloffBonus)


    /**
     * The bonus (or penalty) to max ship velocity.
     */
    val maxVelocityBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.maxVelocityBonus)


    /**
     * The bonus (or penalty) to capacitor recharge time.
     */
    val capRechargeBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.capRechargeBonus)


    /**
     * The bonus (or penalty) to armor hitpoints.
     */
    val armorHitpointsBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.armorHpBonus)


    /**
     * The bonus (or penalty) to shield hitpoints.
     */
    val shieldHitpointsBonus: AttributeProperty<Double>?
        get() = propertyOrNull(attributes.shieldHpBonus)


}