package theorycrafter.ui

import eve.data.*


/**
 * The common implant name prefixes.
 */
private val COMMON_IMPLANT_NAME_PREFIXES = listOf(
    "Inherent Implants 'Noble'",
    "Inherent Implants 'Squire'",
    "Inherent Implants 'Lancer'",
    "Inherent Implants 'Highwall'",
    "Inherent Implants 'Yeti'",
    "Eifyr and Co. 'Gunslinger'",
    "Eifyr and Co. 'Rogue'",
    "Eifyr and Co. 'Alchemist'",
    "Zainou 'Deadeye'",
    "Zainou 'Gnome'",
    "Zainou 'Gypsy'",
    "Zainou 'Beancounter'",
    "Zainou 'Snapshot'",
    "Poteque 'Prospector'",
    "Black Market 'Valdimar'",
    "Inquest 'Eros'",
    "Inquest 'Hedone'",
    "Overmind 'Goliath'",
    "Overmind 'Hawkmoth'",
    "Creodron 'Bumblebee'",
    "Creodron 'Yellowjacket'",
    "Hardwiring - Zainou",
)


/**
 * Shortens implant names by removing the common (and useless) prefixes.
 */
fun ImplantType.shortName(): String {
    val name = this.name
    for (prefix in COMMON_IMPLANT_NAME_PREFIXES){
        if (name.startsWith(prefix))
            return name.substring(prefix.length).trimStart()
    }

    return name
}


/**
 * Common ship substrings and their abbreviations.
 */
private val SHIP_SUBSTRING_ABBREVIATIONS = listOf(
    "Navy Issue" to "NI",
    "Fleet Issue" to "FI",
    "Caldari Navy" to "CN",
    "Federation Navy" to "FN",
    "Imperial Navy" to "IN",
    "Republic Fleet" to "RF",
    "Mark V" to "V",
    "Tribal Issue" to "TI",
    "Federate Issue" to "FI",
    "Imperial Issue" to "II",
    "State Issue" to "SI",
    "Quafe Ultra Edition" to "QUE",
    "Quafe Ultramarine Edition" to "QUME",
    "Amastris Edition" to "AE",
    "Emergency Responder" to "ER",
)


/**
 * Shortens ship names by abbreviating the standard faction names, e.g. "Navy Issue" to "NI".
 */
fun ShipType.shortName(): String {
    val name = this.name
    for ((substring, abbreviation) in SHIP_SUBSTRING_ABBREVIATIONS){
        if (name.contains(substring))
            return name.replace(substring, abbreviation)
    }
    return name
}


/**
 * Shortens tactical mode names by keeping just the kind name.
 */
fun TacticalModeType.shortName(): String {
    return when (kind) {
        TacticalModeType.Kind.DEFENSE -> "Defense"
        TacticalModeType.Kind.PROPULSION -> "Propulsion"
        TacticalModeType.Kind.SHARPSHOOTER -> "Sharpshooter"
    }
}


/**
 * Shortens subsystem names by removing the ship name from them.
 */
fun SubsystemType.shortName(includeKind: Boolean): String {
    return name.substringAfter(if (includeKind) " " else "- ")
}


/**
 * Common ammo substrings and their abbreviations.
 */
private val AMMO_SUBSTRING_ABBREVIATIONS = listOf(
    "Caldari Navy" to "CN",
    "Federation Navy" to "FN",
    "Imperial Navy" to "IN",
    "Republic Fleet" to "RF",
    "Dread Guristas" to "DG",
    "Arch Angel" to "AA",
    "Dark Blood" to "DB",
    "True Sansha" to "TS",
)


/**
 * Shortens ammo names by abbreviating the standard faction names, e.g. "Imperial Navy" to "IN".
 */
fun ChargeType.shortName(): String {
    for ((substring, abbreviation) in AMMO_SUBSTRING_ABBREVIATIONS) {
        if (name.startsWith(substring))
            return abbreviation + name.substring(substring.length)
    }
    return name
}